#include "gdi.h"
#include "bmp.h"

#include <ATI.h>
#include <DAL.h>
#include <mem.h>

#if defined(EMUELF) || defined(WIN32) || defined(GDI_LIBRARY_BUILD)

extern unsigned char default_font_data[256][10];
extern uGDIFont	default_font;


#ifdef GDI_LIBRARY_BUILD
UINT32	_init(void) { return RESULT_OK; }
UINT32	_fini(void)	{ return RESULT_OK; }
#endif


//  ATI
// dCtx			-	   .
// szIdent		-	    (   app_name).
// dMode			-	  .
//				GDI_SYSTEMDC	-    DAL_GetDeviceContext.
//				GDI_PRIVATEDC	-   .
// dDefPatMode	-	  AHIROP.        EndPaint.
// dDrawPatMode	-	  AHIROP   .    GDI_BeginPaint.
// dBufEnabled	-	   
UINT32	GDI_Init(uGDIContext *uCtx, const char *szIdent, void *app, UINT8 dMode, UINT32 dDefPatMode, UINT32 dDrawPatMode, BOOL dBufEnabled) {
	UINT32			result = RESULT_OK;
	AHIDEVICE_T		device;
	AHISURFINFO_T	asinfo;
	AHISURFACE_T	asu;
	AHIPOINT_T		asi = {1, 1};

	//  GDI
	if (uCtx->device.context) { GDI_UnInit(uCtx); }

	uCtx->app = app;
	uCtx->device.state = dMode;
	uCtx->rop.idle = dDefPatMode;
	uCtx->rop.draw = dDrawPatMode;

	uCtx->surfswap = uCtx->rendering = uCtx->painting = false;
	uCtx->dbuffer = dBufEnabled;

	uCtx->region.x1 = uCtx->region.y1 = 0;
	uCtx->region.x2 = DISPLAY_WIDTH;
	uCtx->region.y2 = DISPLAY_HEIGHT;

	uCtx->animation.first = NULL;
	uCtx->animation.last = NULL;
	uCtx->bitmap.first = NULL;
	uCtx->bitmap.last = NULL;

	uCtx->rotate = AHIROT_0;

	switch (dMode) {
		case GDI_SYSTEMDC:
			uCtx->device.context = DAL_GetDeviceContext(0);
			break;
		case GDI_PRIVATEDC:
			device = ldrGetAhiDevice();
			result = AhiDevOpen(&uCtx->device.context, device, szIdent, 0);
			break;
		case GDI_INACTIVE:
		default:
			result = RESULT_FAIL;
	}

	if ((result == RESULT_OK)&&(uCtx->device.context)) {
		uCtx->surfaces.draw.surf = DAL_GetDrawingSurface(DISPLAY_MAIN);
		AhiDispSurfGet(uCtx->device.context, &uCtx->surfaces.disp.surf);
	}

	uCtx->transop.key = ATI_565RGB(255, 0, 255);
	uCtx->transop.mask = 0xFFFF;
	uCtx->transop.cmp = AHICMP_NOTEQUAL;

	//   

	//   
	uCtx->size.x = uCtx->surfaces.disp.size.x = uCtx->surfaces.draw.size.x = DISPLAY_WIDTH;
	uCtx->size.y = uCtx->surfaces.disp.size.y = uCtx->surfaces.draw.size.y = DISPLAY_HEIGHT;

	//    
	AhiSurfAlloc(uCtx->device.context, &asu, &asi, AHIFMT_16BPP_565, AHIFLAG_EXTMEMORY);
	if (asu) {
		AhiSurfFree(uCtx->device.context, asu);
		uCtx->info.extsurf = true;
	} else {
		uCtx->info.extsurf = false;
	}

	//    
	if (uCtx->info.extsurf) { uCtx->flags = GDI_FLAG_SURFACE; } else { uCtx->flags = GDI_FLAG_NOSURF; }

	//  
	GDI_InitFont(uCtx, &default_font);

	return result;
}





//  ATI
UINT32	GDI_UnInit(uGDIContext *uCtx) {
	UINT32		result = RESULT_OK;

	//  
	GDI_UnInitFont(uCtx);

	//  
	GDI_FreeTempSurf(uCtx);

	//  
	while (uCtx->animation.first) {
		GDI_UnLoadAniBitmap(uCtx, uCtx->animation.first);
	}

	//  
	while (uCtx->animation.first) {
		GDI_UnLoadBitmap(uCtx, uCtx->bitmap.first);
	}

	uCtx->app = NULL;

	switch (uCtx->device.state) {
		case GDI_PRIVATEDC:
			if (uCtx->surfswap) {
				GDI_SwapSurfaces(uCtx);
			} else {
				AhiDrawSurfDstSet(uCtx->device.context, uCtx->surfaces.draw.surf, 0);
				AhiDispSurfSet(uCtx->device.context, uCtx->surfaces.disp.surf, 0);
			}
			result = AhiDevClose(uCtx->device.context);
			break;
		default:
			result = RESULT_FAIL;
	}

	uCtx->device.state = GDI_INACTIVE;
	uCtx->device.context = 0;

	UIS_ForceRefresh();

	return result;
}





// ,     
UINT32	GDI_BeginPaint(uGDIContext *uCtx) {
	UINT32		result = RESULT_OK;

	if ((!uCtx->painting)&&(!uCtx->rendering)) {
		switch (uCtx->device.state) {
			case GDI_SYSTEMDC:
			case GDI_PRIVATEDC:
				uCtx->painting = true;
				AhiDrawRopSet(uCtx->device.context, AHIROP3(uCtx->rop.draw));
				AhiDrawSurfDstSet(uCtx->device.context, uCtx->surfaces.draw.surf, 0);
				break;
			default:
				result = RESULT_FAIL;
		}
	} else {
		result = RESULT_FAIL;
	}

	return result;
}





// ,    
UINT32	GDI_EndPaint(uGDIContext *uCtx) {
	UINT32			result = RESULT_OK;

	if (uCtx->painting) { // ||(!uCtx->rendering)) {
		switch (uCtx->device.state) {
			case GDI_SYSTEMDC:
			case GDI_PRIVATEDC:
				//uCtx->rendering = true;
				if (uCtx->dbuffer) {
					GDI_SwapSurfaces(uCtx);
					//AhiDrawSurfDstSet(uCtx->device.context, uCtx->surfaces.draw.surf, 0);
					//AhiDispSurfSet(uCtx->device.context, uCtx->surfaces.disp.surf, 0);
				} else {
					AhiDrawRopSet(uCtx->device.context, AHIROP3(AHIROP_SRCCOPY));
					AhiDrawSurfSrcSet(uCtx->device.context, uCtx->surfaces.draw.surf, 0);
					AhiDrawSurfDstSet(uCtx->device.context, uCtx->surfaces.disp.surf, 0);
					AhiDrawBitBlt(uCtx->device.context, &uCtx->region, (AHIPOINT_T*)&uCtx->region);
				}
				AhiDrawRopSet(uCtx->device.context, AHIROP3(uCtx->rop.idle));
				//uCtx->rendering = false;
				uCtx->painting = false;
				break;
			default:
				result = RESULT_FAIL;
		}
	} else {
		result = RESULT_FAIL;
	}

	return result;
}




//   
UINT32	GDI_SetDisplayRotation(uGDIContext *uCtx, AHIROTATE_T rot, BOOL resupd, BOOL fontupd) {
	UINT32			result = RESULT_OK;
	AHIROTATE_T		prot;

	if (uCtx) {
		//    
		prot = ((rot) ? rot : 4) - uCtx->rotate;
		//    
		uCtx->rotate = rot;
		if (uCtx->rotate % 2) {
			// 90"  270"
			uCtx->size.x = uCtx->region.x2 = DISPLAY_HEIGHT;
			uCtx->size.y = uCtx->region.y2 = DISPLAY_WIDTH;
		} else {
			// 0"  180"
			uCtx->size.x = uCtx->region.x2 = DISPLAY_WIDTH;
			uCtx->size.y = uCtx->region.y2 = DISPLAY_HEIGHT;
		}
		if (resupd) {
			//  
			GDI_RotateAllAniBitmaps(uCtx, prot);
		}
		if (fontupd) {
			//  
			GDI_PrepareFont(uCtx);
		}
	} else {
		result = RESULT_FAIL;
	}

	return result;
}





//    
UINT32 GDI_FillRect(uGDIContext *uCtx, INT32 X, INT32 Y, INT32 W, INT32 H, UINT32 brFg) {
	AHIRECT_T	rect;

	if (brFg != uCtx->transop.key) {
		rect.x1 = X;
		rect.y1 = Y;
		rect.x2 = X + W;
		rect.y2 = Y + H;

		GDI_UpdateRcPos(uCtx, &rect);

		if (rect.x1 > DISPLAY_WIDTH) { rect.x1 = DISPLAY_WIDTH; }
		if (rect.y1 > DISPLAY_HEIGHT) { rect.y1 = DISPLAY_HEIGHT; }
		if (rect.x2 > DISPLAY_WIDTH) { rect.x2 = DISPLAY_WIDTH; }
		if (rect.y2 > DISPLAY_HEIGHT) { rect.y2 = DISPLAY_HEIGHT; }

		AhiDrawRopSet(uCtx->device.context, AHIROP3(AHIROP_PATCOPY));
		AhiDrawBrushFgColorSet(uCtx->device.context, brFg);
		AhiDrawSpans(uCtx->device.context, &rect, 1, AHIFLAG_LINE_STRIP);
		AhiDrawRopSet(uCtx->device.context, AHIROP3(uCtx->rop.draw));
	}

	return RESULT_OK;
}





//    
UINT32 GDI_DrawRect(uGDIContext *uCtx, INT32 X, INT32 Y, INT32 W, INT32 H, UINT32 brFg) {
	AHIPOINT_T	rect[5];

	if (brFg != uCtx->transop.key) {
		rect[0].x = X;
		rect[0].y = Y;
		rect[1].x = X+W;
		rect[1].y = Y;
		rect[2].x = X+W;
		rect[2].y = Y+H;
		rect[3].x = X;
		rect[3].y = Y+H;
		rect[4].x = X;
		rect[4].y = Y;

		GDI_UpdatePtPosMulti(uCtx, rect, 5);

		AhiDrawRopSet(uCtx->device.context, AHIROP3(AHIROP_PATCOPY));
		AhiDrawBrushFgColorSet(uCtx->device.context, brFg);
		AhiDrawLines(uCtx->device.context, rect, 5, AHIFLAG_LINE_STRIP);
		AhiDrawRopSet(uCtx->device.context, AHIROP3(uCtx->rop.draw));
	}

	return RESULT_OK;
}




UINT32	GDI_DrawLine(uGDIContext *uCtx, INT32 X1, INT32 Y1, INT32 X2, INT32 Y2, UINT32 brFg) {
	AHIPOINT_T	line[2];

	if (brFg != uCtx->transop.key) {
		line[0].x = X1;
		line[0].y = Y1;
		line[1].x = X2;
		line[1].y = Y2;

		GDI_UpdatePtPosMulti(uCtx, line, 2);

		AhiDrawRopSet(uCtx->device.context, AHIROP3(AHIROP_PATCOPY));
		AhiDrawBrushFgColorSet(uCtx->device.context, brFg);
		AhiDrawLines(uCtx->device.context, line, 2, AHIFLAG_LINE_STRIP);
		AhiDrawRopSet(uCtx->device.context, AHIROP3(uCtx->rop.draw));
	}

	return RESULT_OK;
}





UINT32  GDI_CreateAniBitmapFromAhiBitmap(uGDIContext *uCtx, uGDIAnimation *uAni, AHIBITMAP_T *hBit) {
	UINT32			result = RESULT_OK;
	AHIPOINT_T		apt = {0, 0};
	AHIRECT_T		arc = {0, 0, 0, 0};

	if ((uAni)&&(hBit->image)) {
		uAni->bitmap.width = hBit->width;
		uAni->bitmap.height = hBit->height;
		uAni->bitmap.image = hBit->image;
		uAni->bitmap.format = hBit->format;
		uAni->bitmap.stride = hBit->stride;
		uAni->height = hBit->height;
		uAni->width = hBit->height;

		uAni->frames = (uAni->bitmap.width)/(uAni->bitmap.height) - 1;

		uAni->timeout = uAni->timestamp = 0;

		uAni->frame = 0;
		if (uAni->frames != 0) { uAni->active = true; } else { uAni->active = false; }

		uAni->frameskip = 0;
		uAni->skipped = 0;

		apt.x = hBit->width;
		apt.y = hBit->height;

		if (uCtx->flags & GDI_FLAG_SURFACE) {
			AhiSurfAlloc(uCtx->device.context, &uAni->surf, &apt, AHIFMT_16BPP_565, AHIFLAG_EXTMEMORY);
			if (uAni->surf) {
				arc.x2 = apt.x;
				arc.y2 = apt.y;
				apt.x = apt.y = arc.x1 = arc.y1 = 0;
				//  
				AhiDrawRopSet(uCtx->device.context, AHIROP3(AHIROP_SRCCOPY));
				AhiDrawSurfDstSet(uCtx->device.context, uAni->surf, 0);
				//   
				AhiDrawBitmapBlt(uCtx->device.context, &arc, &apt, &uAni->bitmap, NULL, 0);
			}
		} else {
			uAni->surf = 0;
		}

		uAni->flags = GDI_BMP_DONTFREE | GDI_ANI_FSW_FRAMES;
		uAni->transparent = false;
		uAni->next = NULL;

		//  
		if (!uCtx->animation.first) { uCtx->animation.first = uAni; }
		if (uCtx->animation.last) { (uCtx->animation.last)->next = uAni; }
		uCtx->animation.last = uAni;

		//    ( )
		if (uCtx->rotate) { GDI_RotateAniBitmap(uCtx, uAni, uCtx->rotate); }
	} else {
		uAni->bitmap.image = NULL;
	}

	return result;
}





UINT32  GDI_CreateAniBitmapFromBinary(uGDIContext *uCtx, uGDIAnimation *uAni, void *bin, UINT32 width, UINT32 height, UINT32 pixfmt) {
	UINT32			result = RESULT_OK;
	AHIPOINT_T		apt = {0, 0};
	AHIRECT_T		arc = {0, 0, 0, 0};
	UINT32			bpp = 1;

	if ((uAni)&&(bin)) {
		switch (pixfmt) {
			case AHIFMT_1BPP:		bpp = 1;	break;
			case AHIFMT_4BPP:		bpp = 4;	break;
			case AHIFMT_8BPP:		bpp = 8;	break;
			case AHIFMT_16BPP_444:
			case AHIFMT_16BPP_555:
			case AHIFMT_16BPP_565:	bpp = 16;	break;
		}
		uAni->bitmap.width = width;
		uAni->bitmap.height = height;
		uAni->bitmap.image = bin;
		uAni->bitmap.format = pixfmt;
		uAni->bitmap.stride = ATI_BITSTRIDE(width, bpp);
		uAni->height = height;
		uAni->width = height;

		uAni->frames = (uAni->bitmap.width)/(uAni->bitmap.height) - 1;

		uAni->timeout = uAni->timestamp = 0;

		uAni->frame = 0;
		if (uAni->frames != 0) { uAni->active = true; } else { uAni->active = false; }

		uAni->frameskip = 0;
		uAni->skipped = 0;

		apt.x = width;
		apt.y = height;

		if (uCtx->flags & GDI_FLAG_SURFACE) {
			AhiSurfAlloc(uCtx->device.context, &uAni->surf, &apt, AHIFMT_16BPP_565, AHIFLAG_EXTMEMORY);
			if (uAni->surf) {
				arc.x2 = apt.x;
				arc.y2 = apt.y;
				apt.x = apt.y = arc.x1 = arc.y1 = 0;
				//  
				AhiDrawRopSet(uCtx->device.context, AHIROP3(AHIROP_SRCCOPY));
				AhiDrawSurfDstSet(uCtx->device.context, uAni->surf, 0);
				//   
				AhiDrawBitmapBlt(uCtx->device.context, &arc, &apt, &uAni->bitmap, NULL, 0);
			}
		} else {
			uAni->surf = 0;
		}

		uAni->flags = GDI_BMP_DONTFREE | GDI_ANI_FSW_FRAMES;
		uAni->transparent = false;
		uAni->next = NULL;

		//  
		if (!uCtx->animation.first) { uCtx->animation.first = uAni; }
		if (uCtx->animation.last) { (uCtx->animation.last)->next = uAni; }
		uCtx->animation.last = uAni;

		//    ( )
		if (uCtx->rotate) { GDI_RotateAniBitmap(uCtx, uAni, uCtx->rotate); }
	} else {
		uAni->bitmap.image = NULL;
	}

	return result;
}





//    BMP    .
UINT32	GDI_LoadAniBitmap(uGDIContext *uCtx, uGDIAnimation *uAni, const WCHAR *szPath) {
	UINT32			result = RESULT_OK;
	AHIPOINT_T		apt = {0, 0};
	AHIRECT_T		arc = {0, 0, 0, 0};

	result = BMP_LoadFromFile(szPath, &uAni->bitmap);

	if (result == RESULT_OK) {
		uAni->width = uAni->bitmap.height;
		uAni->height = uAni->bitmap.height;
		uAni->frames = (uAni->bitmap.width)/(uAni->bitmap.height) - 1;

		uAni->timeout = uAni->timestamp = 0;

		uAni->frame = 0;
		if (uAni->frames != 0) { uAni->active = true; } else { uAni->active = false; }

		uAni->frameskip = 0;
		uAni->skipped = 0;

		apt.x = uAni->bitmap.width;
		apt.y = uAni->bitmap.height;

		if (uCtx->flags & GDI_FLAG_SURFACE) {
			AhiSurfAlloc(uCtx->device.context, &uAni->surf, &apt, AHIFMT_16BPP_565, AHIFLAG_EXTMEMORY);
			if (uAni->surf) {
				arc.x2 = apt.x;
				arc.y2 = apt.y;
				apt.x = apt.y = arc.x1 = arc.y1 = 0;
				//  
				AhiDrawRopSet(uCtx->device.context, AHIROP3(AHIROP_SRCCOPY));
				AhiDrawSurfDstSet(uCtx->device.context, uAni->surf, 0);
				//   
				AhiDrawBitmapBlt(uCtx->device.context, &arc, &apt, &uAni->bitmap, NULL, 0);
			}
		} else {
			uAni->surf = 0;
		}

		uAni->flags = GDI_BMP_FREE | GDI_ANI_FSW_FRAMES;
		uAni->transparent = false;
		uAni->next = NULL;

		//  
		if (!uCtx->animation.first) { uCtx->animation.first = uAni; }
		if (uCtx->animation.last) { (uCtx->animation.last)->next = uAni; }
		uCtx->animation.last = uAni;

		//    ( )
		if (uCtx->rotate) { GDI_RotateAniBitmap(uCtx, uAni, uCtx->rotate); }
	} else {
		uAni->bitmap.image = NULL;
	}

	return result;
}





//    BMP ,    .
UINT32	GDI_UnLoadAniBitmap(uGDIContext *uCtx, uGDIAnimation *uAni) {
	UINT32			result = RESULT_OK;
	uGDIAnimation	*pani = NULL, *ani = uCtx->animation.first;

	if (uAni->bitmap.image != NULL) {
		if (uAni->flags & GDI_BMP_FREE) { suFreeMem(uAni->bitmap.image); }
		uAni->bitmap.image = NULL;

		//  
		if (uAni->surf) {
			AhiSurfFree(uCtx->device.context, uAni->surf);
			uAni->surf = 0;
		}

		//    
		while (ani) {
			if (ani == uAni) {
				if (pani) {
					pani->next = ani->next;
					if (!ani->next) { uCtx->animation.last = pani; }
				} else {
					uCtx->animation.first = ani->next;
					if ((uCtx->animation.first)&&(!(uCtx->animation.first)->next)) {
						uCtx->animation.last = uCtx->animation.first;
					} else { uCtx->animation.last = NULL; }
				}
				break;
			}
			pani = ani;
			ani = ani->next;
		}

	}

	return result;
}






//      
UINT32	GDI_RotateAniBitmapFrame(uGDIContext *uCtx, uGDIAnimation *uAni, AHIROTATE_T rot) {
	UINT32			result = RESULT_OK, zx;

	AHISURFACE_T	asf = uCtx->surfaces.cache.surf;
	AHIRECT_T		arc = {0, 0, 0, 0};
	AHIPOINT_T		apt = {0, 0};

	if (uAni) {
		zx = (uAni->frame)*(uAni->width);
		if ((uAni->surf)&&(uCtx->flags & GDI_FLAG_SURFACE)) {
			//   
			if ((!uCtx->surfaces.cache.surf)||(uCtx->surfaces.cache.size.x < uAni->width)||(uCtx->surfaces.cache.size.y < uAni->height)) {
				apt.x = max(uAni->width, uCtx->surfaces.cache.size.x);
				apt.y = max(uAni->height, uCtx->surfaces.cache.size.y);
				GDI_AllocTempSurf(uCtx, apt.x, apt.y);
			}
			//  
			asf = uAni->surf;
			//  
			apt.x = zx;
			apt.y = 0;
			//    
			arc.x1 = 0;
			arc.y1 = 0;
			arc.x2 = uAni->width;
			arc.y2 = uAni->height;
		} else {
			//      
			if ((!uCtx->surfaces.cache.surf)||(uCtx->surfaces.cache.size.x < 2*uAni->width)||(uCtx->surfaces.cache.size.y < uAni->height)) {
				apt.x = max(2*uAni->width, uCtx->surfaces.cache.size.x);
				apt.y = max(uAni->height, uCtx->surfaces.cache.size.y);
				GDI_AllocTempSurf(uCtx, apt.x, apt.y);
			}
			//  
			asf = uCtx->surfaces.cache.surf;
			//  
			apt.x = 0;
			apt.y = 0;
			//    
			GDI_CopyAniToTempSurf(uCtx, uAni);
			//    
			arc.x1 = uAni->width;
			arc.y1 = 0;
			arc.x2 = 2*uAni->width;
			arc.y2 = uAni->height;
		}
	}

	if (asf) {
		//       
		AhiDrawRopSet(uCtx->device.context, AHIROP3(AHIROP_SRCCOPY));
		AhiDrawSurfSrcSet(uCtx->device.context, asf, 0);
		AhiDrawSurfDstSet(uCtx->device.context, uCtx->surfaces.cache.surf, 0);

		// /
		AhiDrawRotateBlt(uCtx->device.context, &arc, &apt, rot, AHIMIRR_NO, 0);

		//     
		apt.x = arc.x1;
		apt.y = arc.y1;
		arc.x1 = zx;
		arc.y1 = 0;
		arc.x2 = zx + uAni->width;
		arc.y2 = uAni->height;

		//   
		AhiSurfCopy(uCtx->device.context, uCtx->surfaces.cache.surf, &uAni->bitmap, &arc, &apt, 0, AHIFLAG_COPYFROM);

		//    ( )
		if (uAni->surf) {
			AhiDrawSurfSrcSet(uCtx->device.context, uCtx->surfaces.cache.surf, 0);
			AhiDrawSurfDstSet(uCtx->device.context, uAni->surf, 0);

			AhiDrawBitBlt(uCtx->device.context, &arc, &apt);
		}
	} else {
		result = RESULT_FAIL;
	}

	return result;
}






//      
UINT32	GDI_RotateAniBitmap(uGDIContext *uCtx, uGDIAnimation *uAni, AHIROTATE_T rot) {
	UINT32			result = RESULT_OK;
	INT32			i, frame;

	if (uAni) {
		frame = uAni->frame;
		for (i = 0; i <= uAni->frames; i++) {
			uAni->frame = i;
			GDI_RotateAniBitmapFrame(uCtx, uAni, rot);
		}
		uAni->frame = frame;
	}

	return result;
}






//     
UINT32	GDI_RotateAllAniBitmaps(uGDIContext *uCtx, AHIROTATE_T rot) {
	UINT32			result = RESULT_OK;
	uGDIAnimation	*uAni = uCtx->animation.first;

	while (uAni) {
		GDI_RotateAniBitmap(uCtx, uAni, rot);
		uAni = uAni->next;
	}

	return result;
}





UINT32  GDI_CreateBitmapFromAhiBitmap(uGDIContext *uCtx, uGDIBitmap *uBit, AHIBITMAP_T *hBit) {
	UINT32			result = RESULT_OK;
	AHIPOINT_T		apt = {0, 0};
	AHIRECT_T		arc = {0, 0, 0, 0};

	if ((uBit)&&(hBit)) {
		uBit->bitmap.width = hBit->width;
		uBit->bitmap.height = hBit->height;
		uBit->bitmap.image = hBit->image;
		uBit->bitmap.format = hBit->format;
		uBit->bitmap.stride = hBit->stride;
		uBit->width = uBit->bitmap.width;
		uBit->height = uBit->bitmap.height;

		apt.x = uBit->bitmap.width;
		apt.y = uBit->bitmap.height;

		if (uCtx->flags & GDI_FLAG_SURFACE) {
			AhiSurfAlloc(uCtx->device.context, &uBit->surf, &apt, AHIFMT_16BPP_565, AHIFLAG_EXTMEMORY);
			if (uBit->surf) {
				arc.x2 = apt.x;
				arc.y2 = apt.y;
				apt.x = apt.y = arc.x1 = arc.y1 = 0;
				//  
				AhiDrawRopSet(uCtx->device.context, AHIROP3(AHIROP_SRCCOPY));
				AhiDrawSurfDstSet(uCtx->device.context, uBit->surf, 0);
				//   
				AhiDrawBitmapBlt(uCtx->device.context, &arc, &apt, &uBit->bitmap, NULL, 0);
			}
		} else {
			uBit->surf = 0;
		}

		uBit->flags = GDI_BMP_DONTFREE;
		uBit->transparent = false;
		uBit->next = NULL;

		//  
		if (!uCtx->bitmap.first) { uCtx->bitmap.first = uBit; }
		if (uCtx->bitmap.last) { (uCtx->bitmap.last)->next = uBit; }
		uCtx->bitmap.last = uBit;

		//    ( )
		if (uCtx->rotate) { GDI_RotateBitmap(uCtx, uBit, uCtx->rotate); }
	} else {
		uBit->bitmap.image = NULL;
	}

	return result;
}





UINT32  GDI_CreateBitmapFromBinary(uGDIContext *uCtx, uGDIBitmap *uBit, void *bin, UINT32 width, UINT32 height, UINT32 pixfmt) {
	UINT32			result = RESULT_OK;
	AHIPOINT_T		apt = {0, 0};
	AHIRECT_T		arc = {0, 0, 0, 0};
	UINT32			bpp = 1;

	if ((uBit)&&(bin)) {
		switch (pixfmt) {
			case AHIFMT_1BPP:		bpp = 1;	break;
			case AHIFMT_4BPP:		bpp = 4;	break;
			case AHIFMT_8BPP:		bpp = 8;	break;
			case AHIFMT_16BPP_444:
			case AHIFMT_16BPP_555:
			case AHIFMT_16BPP_565:	bpp = 16;	break;
		}
		uBit->bitmap.width = width;
		uBit->bitmap.height = height;
		uBit->bitmap.image = bin;
		uBit->bitmap.format = pixfmt;
		uBit->bitmap.stride = ATI_BITSTRIDE(width, bpp);
		uBit->width = width;
		uBit->height = height;

		apt.x = width;
		apt.y = height;

		if (uCtx->flags & GDI_FLAG_SURFACE) {
			AhiSurfAlloc(uCtx->device.context, &uBit->surf, &apt, AHIFMT_16BPP_565, AHIFLAG_EXTMEMORY);
			if (uBit->surf) {
				arc.x2 = apt.x;
				arc.y2 = apt.y;
				apt.x = apt.y = arc.x1 = arc.y1 = 0;
				//  
				AhiDrawRopSet(uCtx->device.context, AHIROP3(AHIROP_SRCCOPY));
				AhiDrawSurfDstSet(uCtx->device.context, uBit->surf, 0);
				//   
				AhiDrawBitmapBlt(uCtx->device.context, &arc, &apt, &uBit->bitmap, NULL, 0);
			}
		} else {
			uBit->surf = 0;
		}

		uBit->flags = GDI_BMP_DONTFREE;
		uBit->transparent = false;
		uBit->next = NULL;

		//  
		if (!uCtx->bitmap.first) { uCtx->bitmap.first = uBit; }
		if (uCtx->bitmap.last) { (uCtx->bitmap.last)->next = uBit; }
		uCtx->bitmap.last = uBit;

		//    ( )
		if (uCtx->rotate) { GDI_RotateBitmap(uCtx, uBit, uCtx->rotate); }
	} else {
		uBit->bitmap.image = NULL;
	}

	return result;
}





UINT32	GDI_LoadBitmap(uGDIContext *uCtx, uGDIBitmap *uBit, const WCHAR *szPath) {
	UINT32			result = RESULT_OK;
	AHIPOINT_T		apt = {0, 0};
	AHIRECT_T		arc = {0, 0, 0, 0};

	result = BMP_LoadFromFile(szPath, &uBit->bitmap);

	if (result == RESULT_OK) {
		uBit->width = uBit->bitmap.width;
		uBit->height = uBit->bitmap.height;

		apt.x = uBit->bitmap.width;
		apt.y = uBit->bitmap.height;

		if (uCtx->flags & GDI_FLAG_SURFACE) {
			AhiSurfAlloc(uCtx->device.context, &uBit->surf, &apt, AHIFMT_16BPP_565, AHIFLAG_EXTMEMORY);
			if (uBit->surf) {
				arc.x2 = apt.x;
				arc.y2 = apt.y;
				apt.x = apt.y = arc.x1 = arc.y1 = 0;
				//  
				AhiDrawRopSet(uCtx->device.context, AHIROP3(AHIROP_SRCCOPY));
				AhiDrawSurfDstSet(uCtx->device.context, uBit->surf, 0);
				//   
				AhiDrawBitmapBlt(uCtx->device.context, &arc, &apt, &uBit->bitmap, NULL, 0);
			}
		} else {
			uBit->surf = 0;
		}

		uBit->flags = GDI_BMP_FREE;
		uBit->transparent = false;
		uBit->next = NULL;

		//  
		if (!uCtx->bitmap.first) { uCtx->bitmap.first = uBit; }
		if (uCtx->bitmap.last) { (uCtx->bitmap.last)->next = uBit; }
		uCtx->bitmap.last = uBit;

		//    ( )
		if (uCtx->rotate) { GDI_RotateBitmap(uCtx, uBit, uCtx->rotate); }
	} else {
		uBit->bitmap.image = NULL;
	}

	return result;
}






UINT32	GDI_UnLoadBitmap(uGDIContext *uCtx, uGDIBitmap *uBit) {
	UINT32			result = RESULT_OK;
	uGDIBitmap		*pbmp = NULL, *bmp = uCtx->bitmap.first;

	if (uBit->bitmap.image) {
		if (uBit->flags & GDI_BMP_FREE) { suFreeMem(uBit->bitmap.image); }
		uBit->bitmap.image = NULL;

		uBit->bitmap.width = uBit->bitmap.height = uBit->bitmap.format = uBit->bitmap.stride = 0;
		uBit->width = uBit->height = 0;
		uBit->transparent = false;

		//  
		if (uBit->surf) {
			AhiSurfFree(uCtx->device.context, uBit->surf);
			uBit->surf = 0;
		}

		//    
		while (bmp) {
			if (bmp == uBit) {
				if (pbmp) {
					pbmp->next = bmp->next;
					if (!bmp->next) { uCtx->bitmap.last = pbmp; }
				} else {
					uCtx->bitmap.first = bmp->next;
					if ((uCtx->bitmap.first)&&(!(uCtx->bitmap.first)->next)) {
						uCtx->bitmap.last = uCtx->bitmap.first;
					} else { uCtx->bitmap.last = NULL; }
				}
				break;
			}
			pbmp = bmp;
			bmp = bmp->next;
		}
	}

	return result;
}






//    
UINT32	GDI_RotateBitmap(uGDIContext *uCtx, uGDIBitmap *uBit, AHIROTATE_T rot) {
	UINT32			result = RESULT_OK;

	AHISURFACE_T	asf = 0;
	AHIRECT_T		arc = {0, 0, 0, 0};
	AHIPOINT_T		apt = {0, 0};

	if ((uBit)&&(uBit->surf)&&(uCtx->flags & GDI_FLAG_SURFACE)) {
		//  
		if (uCtx->rotate % 2) { apt.x = uBit->height; apt.y = uBit->width; } else { apt.x = uBit->width; apt.y = uBit->height; }

		//    
		AhiSurfAlloc(uCtx->device.context, &asf, &apt, AHIFMT_16BPP_565, AHIFLAG_EXTMEMORY);

		//   
		if (asf) {
			//       
			AhiDrawRopSet(uCtx->device.context, AHIROP3(AHIROP_SRCCOPY));
			AhiDrawSurfSrcSet(uCtx->device.context, uBit->surf, 0);
			AhiDrawSurfDstSet(uCtx->device.context, asf, 0);

			//    
			arc.x1 = 0;
			arc.y1 = 0;
			arc.x2 = apt.x;
			arc.y2 = apt.y;

			//  
			apt.x = 0;
			apt.y = 0;

			//   
			AhiDrawRotateBlt(uCtx->device.context, &arc, &apt, rot, AHIMIRR_NO, 0);

			//        
			AhiSurfFree(uCtx->device.context, uBit->surf);
			uBit->surf = asf;

			//  
			suFreeMem(uBit->bitmap.image);
			uBit->bitmap.width = arc.x2;
			uBit->bitmap.height = arc.y2;
			uBit->bitmap.format = AHIFMT_16BPP_565;
			uBit->bitmap.stride = (uBit->bitmap.width*2 + 3) & (~3);
			uBit->bitmap.image = suAllocMem(uBit->bitmap.stride * uBit->bitmap.height, NULL);
			uBit->transparent = false;

			//   
			AhiSurfCopy(uCtx->device.context, uBit->surf, &uBit->bitmap, &arc, &apt, 0, AHIFLAG_COPYFROM);
		}
	} else {
		result = RESULT_FAIL;
	}

	return result;
}






//     
UINT32	GDI_RotateAllBitmaps(uGDIContext *uCtx, AHIROTATE_T rot) {
	UINT32		result = RESULT_OK;
	uGDIBitmap	*uBit = uCtx->bitmap.first;

	while (uBit) {
		GDI_RotateBitmap(uCtx, uBit, rot);
		uBit = uBit->next;
	}

	return result;
}






UINT32	GDI_RenderRegionToBitmap(uGDIContext *uCtx, uGDIBitmap *uBit, BOOL disp) {
	UINT32			result = RESULT_OK;

	AHIPOINT_T		apt = {uCtx->region.x1, uCtx->region.y1};
	AHIRECT_T		arc = {0, 0, uCtx->region.x2 - uCtx->region.x1 + 1, uCtx->region.y2 - uCtx->region.y1 + 1};
	AHISURFACE_T	asf = (disp) ? uCtx->surfaces.disp.surf : uCtx->surfaces.draw.surf;

	if (uBit) {
		GDI_UnLoadBitmap(uCtx, uBit);

		uBit->width = uBit->bitmap.width = arc.x2;
		uBit->height = uBit->bitmap.height = arc.y2;
		uBit->bitmap.format = AHIFMT_16BPP_565;
		uBit->bitmap.stride = (uBit->bitmap.width*2 + 3) & (~3);
		uBit->bitmap.image = suAllocMem(uBit->bitmap.stride * uBit->bitmap.height, NULL);
		uBit->transparent = false;
		uBit->flags = GDI_BMP_FREE;

		AhiDrawRopSet(uCtx->device.context, AHIROP3(AHIROP_SRCCOPY));

		AhiSurfCopy(uCtx->device.context, asf, &uBit->bitmap, &arc, &apt, 0, AHIFLAG_COPYFROM);
	} else {
		result = RESULT_FAIL;
	}

	return result;
}






UINT32	GDI_RenderBitmap(uGDIContext *uCtx, uGDIBitmap *uBit, INT32 X, INT32 Y) {
	UINT32			result = RESULT_OK;

	AHIPOINT_T		apt = {0, 0};
	AHIRECT_T		arc = {X, Y, 0, 0};

	if (uBit) {
		arc.x2 = X + uBit->width;
		arc.y2 = Y + uBit->height;

		GDI_UpdateRcPos(uCtx, &arc);

		//       
		if (arc.x1 < 0) { apt.x -= arc.x1; arc.x1 = 0; }
		if (arc.y1 < 0) { apt.y -= arc.y1; arc.y1 = 0; }
		if (arc.x1 > DISPLAY_WIDTH) { arc.x1 = DISPLAY_WIDTH; }
		if (arc.y1 > DISPLAY_HEIGHT) { arc.y1 = DISPLAY_HEIGHT; }
		if (arc.x2 > DISPLAY_WIDTH) { arc.x2 = DISPLAY_WIDTH; }
		if (arc.y2 > DISPLAY_HEIGHT) { arc.y2 = DISPLAY_HEIGHT; }

		AhiDrawRopSet(uCtx->device.context, AHIROP3(AHIROP_SRCCOPY));
		AhiDrawSurfDstSet(uCtx->device.context, uCtx->surfaces.draw.surf, 0);

		if ((uBit->surf)&&(uCtx->flags & GDI_FLAG_SURFACE)) {
			AhiDrawSurfSrcSet(uCtx->device.context, uBit->surf, 0);

			AhiDrawBitBlt(uCtx->device.context, &arc, &apt);
		} else {
			AhiDrawBitmapBlt(uCtx->device.context, &arc, &apt, &uBit->bitmap, NULL, 0);
		}
	}

	return result;
}






//    BMP .
UINT32	GDI_RenderAniBitmap(uGDIContext *uCtx, uGDIAnimation *uAni, INT32 X, INT32 Y) {
	UINT32			result = RESULT_OK;

	AHISURFACE_T	asf = uCtx->surfaces.cache.surf;
	AHIRECT_T		arc = {X, Y, 0, 0};
	AHIPOINT_T		apt = {0, 0};

	if (uAni) {
		if ((uAni->surf)&&(uCtx->flags & GDI_FLAG_SURFACE)) {
			//  
			asf = uAni->surf;
			//  
			apt.x = (uAni->frame)*(uAni->width);
			apt.y = 0;
			//    
			arc.x2 = X+uAni->width;
			arc.y2 = Y+uAni->height;
		} else {
			//    
			GDI_CopyAniToTempSurf(uCtx, uAni);
			//    
			arc.x2 = X+uCtx->anisize.x;
			arc.y2 = Y+uCtx->anisize.y;
		}
		//    
		if (uAni->active) { GDI_NextAniFrame(uAni); }
	}

	if (asf) {
		//       
		AhiDrawRopSet(uCtx->device.context, AHIROP3(AHIROP_SRCCOPY));
		AhiDrawSurfSrcSet(uCtx->device.context, asf, 0);
		AhiDrawSurfDstSet(uCtx->device.context, uCtx->surfaces.draw.surf, 0);

		//      
		GDI_UpdateRcPos(uCtx, &arc);

		//       
		if (arc.x1 > DISPLAY_WIDTH) { arc.x1 = DISPLAY_WIDTH; }
		if (arc.y1 > DISPLAY_HEIGHT) { arc.y1 = DISPLAY_HEIGHT; }
		if (arc.x2 > DISPLAY_WIDTH) { arc.x2 = DISPLAY_WIDTH; }
		if (arc.y2 > DISPLAY_HEIGHT) { arc.y2 = DISPLAY_HEIGHT; }

		if ((arc.x1 < arc.x2)&&(arc.y1 < arc.y2)) {
			//   
			AhiDrawBitBlt(uCtx->device.context, &arc, &apt);
		}

		//   
		AhiDrawRopSet(uCtx->device.context, AHIROP3(uCtx->rop.draw));
	} else {
		result = RESULT_FAIL;
	}

	return result;
}






UINT32	GDI_RenderAniBitmapMulti(uGDIContext *uCtx, uGDIAnimation *uAni, AHIRECT_T *dstRectArr, AHIPOINT_T *srcPtArr, UINT32 count) {
	UINT32			result = RESULT_OK, i;

	AHISURFACE_T	asf = uCtx->surfaces.cache.surf;

	if (uAni) {
		if ((uAni->surf)&&(uCtx->flags & GDI_FLAG_SURFACE)) {
			//  
			asf = uAni->surf;
		} else {
			//    
			GDI_CopyAniToTempSurf(uCtx, uAni);
		}
		//    
		if (uAni->active) { GDI_NextAniFrame(uAni); }
	}

	if ((asf)&&(count)) {
		//      
		GDI_UpdateRcPosMulti(uCtx, dstRectArr, count);
		//      
		for (i = 0; i < count; i++) {
			if (dstRectArr[i].x1 > DISPLAY_WIDTH) { dstRectArr[i].x1 = DISPLAY_WIDTH; }
			if (dstRectArr[i].y1 > DISPLAY_HEIGHT) { dstRectArr[i].y1 = DISPLAY_HEIGHT; }
			if (dstRectArr[i].x2 > DISPLAY_WIDTH) { dstRectArr[i].x2 = DISPLAY_WIDTH; }
			if (dstRectArr[i].y2 > DISPLAY_HEIGHT) { dstRectArr[i].y2 = DISPLAY_HEIGHT; }
		}

		//       
		AhiDrawRopSet(uCtx->device.context, AHIROP3(AHIROP_SRCCOPY));
		AhiDrawSurfSrcSet(uCtx->device.context, asf, 0);
		AhiDrawSurfDstSet(uCtx->device.context, uCtx->surfaces.draw.surf, 0);

		//   
		AhiDrawBitBltMulti(uCtx->device.context, dstRectArr, srcPtArr, count);

		//   
		AhiDrawRopSet(uCtx->device.context, AHIROP3(uCtx->rop.draw));
	} else {
		result = RESULT_FAIL;
	}

	return result;
}





//      BMP  (       120x64)
UINT32	GDI_RenderAniBitmapTrans(uGDIContext *uCtx, uGDIAnimation *uAni, INT32 X, INT32 Y) {
	UINT32			result = RESULT_OK;
	UINT32			i, j;

	AHISURFACE_T	asf = uCtx->surfaces.cache.surf;
	AHIRECT_T		arc = {X, Y, 0, 0};
	AHIPOINT_T		apt = {0, 0};

	if (uAni) {
		if ((uAni->surf)&&(uCtx->flags & GDI_FLAG_SURFACE)) {
			//  
			asf = uAni->surf;
			//  
			apt.x = (uAni->frame)*(uAni->width);
			apt.y = 0;
			//    
			arc.x2 = X+uAni->width;
			arc.y2 = Y+uAni->height;
		} else {
			//    
			GDI_CopyAniToTempSurf(uCtx, uAni);
			//    
			arc.x2 = X+uCtx->anisize.x;
			arc.y2 = Y+uCtx->anisize.y;
		}
		//    
		if (uAni->active) { GDI_NextAniFrame(uAni); }
	}

	if (asf) {
		//       
		AhiDrawRopSet(uCtx->device.context, AHIROP3(AHIROP_SRCCOPY));
		AhiDrawSurfSrcSet(uCtx->device.context, asf, 0);
		AhiDrawSurfDstSet(uCtx->device.context, uCtx->surfaces.draw.surf, 0);

		//      
		GDI_UpdateRcPos(uCtx, &arc);

		//       
		if (arc.x1 > DISPLAY_WIDTH) { arc.x1 = DISPLAY_WIDTH; }
		if (arc.y1 > DISPLAY_HEIGHT) { arc.y1 = DISPLAY_HEIGHT; }
		if (arc.x2 > DISPLAY_WIDTH) { arc.x2 = DISPLAY_WIDTH; }
		if (arc.y2 > DISPLAY_HEIGHT) { arc.y2 = DISPLAY_HEIGHT; }

		if ((arc.x1 < arc.x2)&&(arc.y1 < arc.y2)) {
			//     
			AhiDrawTransBlt(uCtx->device.context, &arc, &apt, NULL, &uCtx->transop);
		}

		//   
		AhiDrawRopSet(uCtx->device.context, AHIROP3(uCtx->rop.draw));
	} else {
		result = RESULT_FAIL;
	}

	return result;
}






UINT32	GDI_RenderAniBitmapTransMulti(uGDIContext *uCtx, uGDIAnimation *uAni, AHIRECT_T *dstRectArr, AHIPOINT_T *srcPtArr, UINT32 count) {
	UINT32			result = RESULT_OK, i;

	AHISURFACE_T	asf = uCtx->surfaces.cache.surf;

	if (uAni) {
		if ((uAni->surf)&&(uCtx->flags & GDI_FLAG_SURFACE)) {
			//  
			asf = uAni->surf;
		} else {
			//    
			GDI_CopyAniToTempSurf(uCtx, uAni);
		}
		//    
		if (uAni->active) { GDI_NextAniFrame(uAni); }
	}

	if ((asf)&&(count)) {
		//      
		GDI_UpdateRcPosMulti(uCtx, dstRectArr, count);
		//      
		for (i = 0; i < count; i++) {
			if (dstRectArr[i].x1 > DISPLAY_WIDTH) { dstRectArr[i].x1 = DISPLAY_WIDTH; }
			if (dstRectArr[i].y1 > DISPLAY_HEIGHT) { dstRectArr[i].y1 = DISPLAY_HEIGHT; }
			if (dstRectArr[i].x2 > DISPLAY_WIDTH) { dstRectArr[i].x2 = DISPLAY_WIDTH; }
			if (dstRectArr[i].y2 > DISPLAY_HEIGHT) { dstRectArr[i].y2 = DISPLAY_HEIGHT; }
		}

		//       
		AhiDrawRopSet(uCtx->device.context, AHIROP3(AHIROP_SRCCOPY));
		AhiDrawSurfSrcSet(uCtx->device.context, asf, 0);
		AhiDrawSurfDstSet(uCtx->device.context, uCtx->surfaces.draw.surf, 0);

		//     
		for (i = 0; i < count; i++) {
			AhiDrawTransBlt(uCtx->device.context, &dstRectArr[i], &srcPtArr[i], NULL, &uCtx->transop);
		}

		//   
		AhiDrawRopSet(uCtx->device.context, AHIROP3(uCtx->rop.draw));
	} else {
		result = RESULT_FAIL;
	}

	return result;
}





//       
void	GDI_NextAniFrame(uGDIAnimation *uAni) {
	UINT32			st, diff;

	if (uAni->flags & GDI_ANI_FSW_FRAMES) {
		if (uAni->skipped < uAni->frameskip) { uAni->skipped++; } else {
			if (uAni->frame < uAni->frames) { uAni->frame++; } else { uAni->frame = 0; }
			uAni->skipped = 0;
		}
	} else if (uAni->flags & GDI_ANI_FSW_TIME) {
		st = (UINT32)suPalTicksToMsec(suPalReadTime());
		diff = st - uAni->timestamp;
		if (diff > uAni->timeout) {
			if (uAni->frame < uAni->frames) {
				uAni->frame += diff / uAni->timeout;
				uAni->frame %= uAni->frames + 1;
			} else { uAni->frame = 0; }
			uAni->timestamp = st;
		}
	}
}





//       
UINT32	GDI_AllocTempSurf(uGDIContext *uCtx, INT32 width, INT32 height) {
	UINT32			result = RESULT_OK;
	AHIPOINT_T		apt;

	if (uCtx->surfaces.cache.surf) {
		GDI_FreeTempSurf(uCtx);
	}

	uCtx->surfaces.cache.size.x = apt.x = width;
	uCtx->surfaces.cache.size.y = apt.y = height;
	uCtx->surfaces.cache.surf = DAL_GetCachedSurface(apt);

	if (!uCtx->surfaces.cache.surf) {
		uCtx->surfaces.cache.surf = RESULT_FAIL;
	}

	return result;
}





UINT32  GDI_CopyAniToTempSurf(uGDIContext *uCtx, uGDIAnimation *uAni) {
	UINT32			result = RESULT_OK;

	AHIRECT_T		arc;
	AHIPOINT_T		apt;

	if ((!uCtx->surfaces.cache.surf)||(uCtx->surfaces.cache.size.x < uAni->width)||(uCtx->surfaces.cache.size.y < uAni->height)) {
		apt.x = max(uAni->width, uCtx->surfaces.cache.size.x);
		apt.y = max(uAni->height, uCtx->surfaces.cache.size.y);
		GDI_AllocTempSurf(uCtx, apt.x, apt.y);
	}

	if ((uCtx->surfaces.cache.surf)&&(uAni->bitmap.image)) {
		//   
		uCtx->anisize.x = uAni->width;
		uCtx->anisize.y = uAni->height;

		//    
		arc.x1 = arc.y1 = 0;
		arc.x2 = uAni->width;
		arc.y2 = uAni->height;

		//     
		apt.x = (uAni->frame)*(uAni->width);
		apt.y = 0;

		//       
		AhiDrawRopSet(uCtx->device.context, AHIROP3(AHIROP_SRCCOPY));
		AhiDrawSurfDstSet(uCtx->device.context, uCtx->surfaces.cache.surf, 0);

		//  -   
		result = AhiDrawBitmapBlt(uCtx->device.context, &arc, &apt, &uAni->bitmap, NULL, 0);
		if (result) { uCtx->surfaces.cache.surf = 0; }

		//   
		AhiDrawRopSet(uCtx->device.context, AHIROP3(uCtx->rop.draw));
	} else {
		result = RESULT_FAIL;
	}

	return result;
}





//       
UINT32	GDI_FreeTempSurf(uGDIContext *uCtx) {
	UINT32			result = RESULT_OK;

	if (uCtx->surfaces.cache.surf) {
		AhiSurfFree(uCtx->device.context, uCtx->surfaces.cache.surf);

		uCtx->surfaces.cache.surf = NULL;
	} else {
		result = RESULT_FAIL;
	}

	return result;
}






UINT32	GDI_InitFont(uGDIContext *uCtx, uGDIFont *uFont) {
	UINT32		result = RESULT_OK, i, j, k, w, h, size;
	BOOL		f;
	char		*data;
	void		*block;

	uCtx->font.width = uFont->width;
	uCtx->font.height = uFont->height;
	uCtx->font.count = uFont->count;
	uCtx->font.rot = uFont->rot;
	uCtx->font.data = uFont->data;
	
	uCtx->font.chars = NULL;
	uCtx->font.rect = NULL;
	uCtx->font.size = NULL;

	if (uCtx->font.data) {
		//    
		block = suAllocMem(uCtx->font.count * (sizeof(AHIRECT_T) + sizeof(AHIPOINT_T)), NULL);
		uCtx->font.rect = block;
		uCtx->font.size = block + (uCtx->font.count * sizeof(AHIRECT_T));
		//  
		w = uCtx->font.width;
		h = uCtx->font.height;
		size = w * h;
		for (i = 0; i < uCtx->font.count; i++) {
			//       
			data = uCtx->font.data + (i * size)/8;
			//     
			uCtx->font.rect[i].x1 = 0;
			uCtx->font.rect[i].y1 = 0;
			uCtx->font.rect[i].x2 = 4;
			uCtx->font.rect[i].y2 = h - 1;
			//   
			f = false;
			for (j = 0; (j < w) && (!f); j++) {
				for (k = 0; (k < h) && (!f); k++) {
					if (_bit(data, (k * w) + j)) { uCtx->font.rect[i].x1 = j; f = true; }
				}
			}
			//   
			f = false;
			for (j = w - 1; ((j + 1) > 0) && (!f); j--) {
				for (k = 0; (k < h) && (!f); k++) {
					if (_bit(data, (k * w) + j)) { uCtx->font.rect[i].x2 = j; f = true; }
				}
			}
			//   
			f = false;
			for (j = 0; (j < h) && (!f); j++) {
				for (k = 0; (k < w) && (!f); k++) {
					if (_bit(data, (j * w) + k)) { uCtx->font.rect[i].y1 = j; f = true; }
				}
			}
			//   
			f = false;
			for (j = h - 1; ((j + 1) > 0) && (!f); j--) {
				for (k = 0; (k < w) && (!f); k++) {
					if (_bit(data, (j * w) + k)) { uCtx->font.rect[i].y2 = j; f = true; }
				}
			}
			//     
			uCtx->font.size[i].x = uCtx->font.rect[i].x2 - uCtx->font.rect[i].x1 + 1;
			uCtx->font.size[i].y = uCtx->font.rect[i].y2 - uCtx->font.rect[i].y1 + 1;
		}
		//   
		GDI_PrepareFont(uCtx);
	}

	return result;
}





UINT32	GDI_PrepareFont(uGDIContext *uCtx) {
	UINT32		result = RESULT_OK, i;
	char		*data;
	AHIBITMAP_T	bmp;

	if (uCtx->font.data) {
		//     
		if (uCtx->rotate % 2) {
			bmp.width = uCtx->font.height;
			bmp.height = uCtx->font.width;
		} else {
			bmp.width = uCtx->font.width;
			bmp.height = uCtx->font.height;
		}
#ifndef WIN32
		bmp.format = AHIFMT_1BPP;
		bmp.stride = ATI_BITSTRIDE(bmp.width, 1);
#else
		bmp.format = AHIFMT_16BPP_565;
		bmp.stride = ATI_BITSTRIDE(bmp.width, 16);
#endif
		//      
		if (uCtx->font.chars) { suFreeMem(uCtx->font.chars); }
		uCtx->font.chars = suAllocMem(uCtx->font.count * bmp.height * bmp.stride, NULL);
		//   ,       
		if (uCtx->font.chars) {
			for (i = 0; i < uCtx->font.count; i++) {
				//       
				data = uCtx->font.data + (i * uCtx->font.width * uCtx->font.height)/8;
				bmp.image = (void*)(uCtx->font.chars + (i * (bmp.height * bmp.stride)));
				//  
				AhiUnpackBitmap(&bmp, data, uCtx->font.width, uCtx->font.height, uCtx->rotate);
			}
		}
	}

	return result;
}





UINT32	GDI_UnInitFont(uGDIContext *uCtx) {
	UINT32		result = RESULT_OK;

	if (uCtx->font.chars)	{ suFreeMem(uCtx->font.chars);		uCtx->font.chars = NULL;	}
	if (uCtx->font.rect)	{ suFreeMem(uCtx->font.rect);
		uCtx->font.rect = NULL;
		uCtx->font.size = NULL;
	}

	return result;
}












UINT32	GDI_RenderString(uGDIContext *uCtx, char *string, INT32 X, INT32 Y, UINT32 brFg, UINT32 distance) {
	UINT32		result = RESULT_OK, i = 0;
	INT32		stride;

	AHIBITMAP_T	symbol;
	AHIPOINT_T	spt;
	AHIRECT_T	drc;

	AhiDrawRopSet(uCtx->device.context, AHIROP3(AHIROP_SRCCOPY));
	AhiDrawFgColorSet(uCtx->device.context, brFg);

	while ((*string) && (i < 256)) {
		if (uCtx->rotate % 2) {
			symbol.width = uCtx->font.height;
			symbol.height = uCtx->font.width;
		} else {
			symbol.width = uCtx->font.width;
			symbol.height = uCtx->font.height;
		}
#ifndef WIN32
		symbol.format = AHIFMT_1BPP;
		symbol.stride = ATI_BITSTRIDE(symbol.width, 1);
#else
		symbol.format = AHIFMT_16BPP_565;
		symbol.stride = ATI_BITSTRIDE(symbol.width, 16);
#endif
		symbol.image = (void*)(uCtx->font.chars + (*string * (symbol.height * symbol.stride)));
		if (i) {
			//  
			stride += uCtx->font.size[*(string - 1)].x + distance;
		} else {
			//      
			stride = 0;
		}
		//    
		switch (uCtx->rotate) {
			case AHIROT_0:
				spt.x = uCtx->font.rect[*(string)].x1;
				spt.y = uCtx->font.rect[*(string)].y1;
				break;
			case AHIROT_90:
				spt.x = uCtx->font.height - uCtx->font.rect[*(string)].y2 - 1;
				spt.y = uCtx->font.rect[*(string)].x1;
				break;
			case AHIROT_180:
				spt.x = uCtx->font.width - uCtx->font.rect[*(string)].x2 - 1;
				spt.y = uCtx->font.height - uCtx->font.rect[*(string)].y2 - 1;
				break;
			case AHIROT_270:
				spt.x = uCtx->font.rect[*(string)].y1;
				spt.y = uCtx->font.width - uCtx->font.rect[*(string)].x2 - 1;
				break;
		}
		//   
		drc.x1 = X + stride;
		drc.y1 = Y + uCtx->font.rect[*(string)].y1;
		drc.x2 = drc.x1 + uCtx->font.size[*(string)].x;
		drc.y2 = drc.y1 + uCtx->font.size[*(string)].y;
		//  
		GDI_UpdateRcPos(uCtx, &drc);
		//  
#ifndef WIN32
		result = AhiDrawBitmapBlt(uCtx->device.context, &drc, &spt, &symbol, NULL, 1);
#else
		result = AhiDrawBitmapBlt(uCtx->device.context, &drc, &spt, &symbol, NULL, 0);
#endif
		//    
		string++;
		i++;
	}

	AhiDrawRopSet(uCtx->device.context, AHIROP3(uCtx->rop.draw));

	return result;
}






UINT32	GDI_GetStringWidth(uGDIContext *uCtx, char *string, UINT32 distance) {
	UINT32		result = 0;

	while (*string) {
		result += uCtx->font.size[*string].x + distance;
		string++;
	}

	return result - distance;
}






UINT32	GDI_GetStringHeight(uGDIContext *uCtx, char *string) {
	INT32		result = 0;

	while (*string) {
		if (uCtx->font.size[*string].y > result) { result = uCtx->font.size[*string].y; }
		string++;
	}

	return (UINT32)result;
}

#endif		// if defined(EMUELF) || defined(WIN32) || defined(GDI_LIBRARY_BUILD)





void	GDI_SwapSurfaces(uGDIContext *uCtx) {
	UINT32		surf;

	uCtx->surfswap = !uCtx->surfswap;

	surf = uCtx->surfaces.disp.surf;
	uCtx->surfaces.disp.surf = uCtx->surfaces.draw.surf;
	uCtx->surfaces.draw.surf = surf;

	AhiDrawSurfDstSet(uCtx->device.context, uCtx->surfaces.draw.surf, 0);
	AhiDispSurfSet(uCtx->device.context, uCtx->surfaces.disp.surf, 0);
}





//            
void	GDI_UpdatePtPos(uGDIContext *uCtx, AHIPOINT_T *pt) {
	INT32	z;

	switch (uCtx->rotate) {
		case AHIROT_90:
			z = pt->x;
			pt->x = DISPLAY_WIDTH - pt->y;
			pt->y = z;
			break;
		case AHIROT_180:
			pt->x = DISPLAY_WIDTH - pt->x;
			pt->y = DISPLAY_HEIGHT - pt->y;
			break;
		case AHIROT_270:
			z = pt->x;
			pt->x = pt->y;
			pt->y = DISPLAY_HEIGHT - z;
			break;
	}
}

void	GDI_UpdatePtPosMulti(uGDIContext *uCtx, AHIPOINT_T *pt, INT32 count) {
	INT32	i, z;

	for (i = 0; i < count; i++) {
		switch (uCtx->rotate) {
			case AHIROT_90:
				z = pt[i].x;
				pt[i].x = DISPLAY_WIDTH - pt[i].y;
				pt[i].y = z;
				break;
			case AHIROT_180:
				pt[i].x = DISPLAY_WIDTH - pt[i].x;
				pt[i].y = DISPLAY_HEIGHT - pt[i].y;
				break;
			case AHIROT_270:
				z = pt[i].x;
				pt[i].x = pt[i].y;
				pt[i].y = DISPLAY_HEIGHT - z;
				break;
		}
	}
}





//            
void	GDI_UpdateRcPos(uGDIContext *uCtx, AHIRECT_T *rc) {
	INT32	z, w = rc->x2 - rc->x1, h = rc->y2 - rc->y1;

	switch (uCtx->rotate) {
		case AHIROT_90:
			z = rc->x1;
			rc->x1 = DISPLAY_WIDTH - rc->y1 - h;
			rc->y1 = z;
			rc->x2 = rc->x1 + h;
			rc->y2 = rc->y1 + w;
			break;
		case AHIROT_180:
			rc->x1 = DISPLAY_WIDTH - rc->x1 - w;
			rc->y1 = DISPLAY_HEIGHT - rc->y1 - h;
			rc->x2 = rc->x1 + w;
			rc->y2 = rc->y1 + h;
			break;
		case AHIROT_270:
			z = rc->x1;
			rc->x1 = rc->y1;
			rc->y1 = DISPLAY_HEIGHT - z - w;
			rc->x2 = rc->x1 + h;
			rc->y2 = rc->y1 + w;
			break;
	}
}

void	GDI_UpdateRcPosMulti(uGDIContext *uCtx, AHIRECT_T *rc, INT32 count) {
	INT32	i, z, w, h;

	for (i = 0; i < count; i++) {
		w = rc[i].x2 - rc[i].x1;
		h = rc[i].y2 - rc[i].y1;
		switch (uCtx->rotate) {
			case AHIROT_90:
				z = rc[i].x1;
				rc[i].x1 = DISPLAY_WIDTH - rc[i].y1 - h;
				rc[i].y1 = z;
				rc[i].x2 = rc[i].x1 + h;
				rc[i].y2 = rc[i].y1 + w;
				break;
			case AHIROT_180:
				rc[i].x1 = DISPLAY_WIDTH - rc[i].x1 - w;
				rc[i].y1 = DISPLAY_HEIGHT - rc[i].y1 - h;
				rc[i].x2 = rc[i].x1 + w;
				rc[i].y2 = rc[i].y1 + h;
				break;
			case AHIROT_270:
				z = rc[i].x1;
				rc[i].x1 = rc[i].y1;
				rc[i].y1 = DISPLAY_HEIGHT - z - w;
				rc[i].x2 = rc[i].x1 + h;
				rc[i].y2 = rc[i].y1 + w;
				break;
		}
	}
}





void	AhiGetBitmapPixel(AHIBITMAP_T *bitmap, INT32 X, INT32 Y, UINT32 *pixel) {

	switch (bitmap->format) {
		case AHIFMT_1BPP:
			*pixel = (*(UINT8*)(bitmap->image + (Y * bitmap->stride) + (X / 8)) >> (8 - 1 - (X % 8))) & 0x1;
			break;
		case AHIFMT_4BPP:
			*pixel = (*(UINT8*)(bitmap->image + (Y * bitmap->stride) + (X / 2)) >> (4 * (2 - 1 - (X % 2)))) & 0xF;
			break;
		case AHIFMT_8BPP:
			*pixel = *(UINT8*)(bitmap->image + (Y * bitmap->stride) + X) & 0xFF;
			break;
		case AHIFMT_16BPP_444:
		case AHIFMT_16BPP_555:
		case AHIFMT_16BPP_565:
			*pixel = *(UINT16*)(bitmap->image + (Y * bitmap->stride) + (2 * X)) & 0xFFFF;
			break;
	}
}





void	AhiSetBitmapPixel(AHIBITMAP_T *bitmap, INT32 X, INT32 Y, UINT32 pixel) {
	INT32	shift;

	switch (bitmap->format) {
		case AHIFMT_1BPP:
			shift = 8 - 1 - (X % 8);
			*(UINT8*)(bitmap->image + (Y * bitmap->stride) + (X / 8)) &= (UINT8)((0x1 << shift) ^ 0xFF);
			*(UINT8*)(bitmap->image + (Y * bitmap->stride) + (X / 8)) |= (UINT8)((pixel & 0x1) << shift);
			break;
		case AHIFMT_4BPP:
			shift = 4 * (2 - 1 - (X % 2));
			*(UINT8*)(bitmap->image + (Y * bitmap->stride) + (X / 2)) &= (UINT8)((0xF << shift) ^ 0xFF);
			*(UINT8*)(bitmap->image + (Y * bitmap->stride) + (X / 2)) |= (UINT8)((pixel & 0xF) << shift);
			break;
		case AHIFMT_8BPP:
			*(UINT8*)(bitmap->image + (Y * bitmap->stride) + X) = (UINT8)(pixel);
			break;
		case AHIFMT_16BPP_444:
		case AHIFMT_16BPP_555:
		case AHIFMT_16BPP_565:
			*(UINT16*)(bitmap->image + (Y * bitmap->stride) + (2 * X)) = (UINT16)(pixel);
			break;
	}
}





void	AhiUnpackBitmap(AHIBITMAP_T *bitmap, void *data, INT32 width, INT32 height, AHIROTATE_T rot) {
	UINT32		i, len = abs(width * height);

	if ((bitmap) && (data)) {
		//    
#ifndef WIN32
		//      
		switch (rot) {
			case AHIROT_0:
				for (i = 0; i < len; i++) { AhiSetBitmapPixel(bitmap, (i % width), (i / width), _bit(data, i)); }
				break;
			case AHIROT_90:
				for (i = 0; i < len; i++) { AhiSetBitmapPixel(bitmap, (height - (i / width) - 1), (i % width), _bit(data, i)); }
				break;
			case AHIROT_180:
				for (i = 0; i < len; i++) { AhiSetBitmapPixel(bitmap, i % width, i / width, _bit(data, len - i - 1)); }
				break;
			case AHIROT_270:
				for (i = 0; i < len; i++) { AhiSetBitmapPixel(bitmap, (i / width), (width - (i % width) - 1), _bit(data, i)); }
				break;
		}
#else
		//     16- 
		switch (rot) {
			case AHIROT_0:
				for (i = 0; i < len; i++) { AhiSetBitmapPixel(bitmap, i % width, i / width, (_bit(data, i)) ? 0xFFFF : 0); }
				break;
			case AHIROT_90:
				for (i = 0; i < len; i++) { AhiSetBitmapPixel(bitmap, (height - (i / width) - 1), (i % width), (_bit(data, i)) ? 0xFFFF : 0); }
				break;
			case AHIROT_180:
				for (i = 0; i < len; i++) { AhiSetBitmapPixel(bitmap, i % width, i / width, (_bit(data, len - i - 1)) ? 0xFFFF : 0); }
				break;
			case AHIROT_270:
				for (i = 0; i < len; i++) { AhiSetBitmapPixel(bitmap, (i / width), (width - (i % width) - 1), (_bit(data, i)) ? 0xFFFF : 0); }
				break;
		}
#endif
	}
}
